USE [master]
GO


PRINT N'Creating database [SCF_Invoices]';
CREATE DATABASE [SCF_Invoices]
GO

USE [SCF_Invoices];


GO
PRINT N'Creating [dbo].[DI_Bucket]...';


GO
CREATE TABLE [dbo].[DI_Bucket] (
    [BucketID]       INT          IDENTITY (1, 1) NOT NULL,
    [BucketName]     VARCHAR (50) NOT NULL,
    [BucketFromDays] INT          NOT NULL,
    [BucketToDays]   INT          NOT NULL,
    PRIMARY KEY CLUSTERED ([BucketID] ASC)
);


GO
PRINT N'Creating [dbo].[DI_Customer]...';


GO
CREATE TABLE [dbo].[DI_Customer] (
    [CustomerID]      INT           IDENTITY (1, 1) NOT NULL,
    [CustomerIDO]     INT           NOT NULL,
    [CustomerName]    VARCHAR (50)  NOT NULL,
    [CustomerAddress] VARCHAR (500) NOT NULL,
    PRIMARY KEY CLUSTERED ([CustomerID] ASC)
);


GO
PRINT N'Creating [dbo].[SCF_Invoices]...';


GO
CREATE TABLE [dbo].[SCF_Invoices] (
    [InvoiceID]  INT             IDENTITY (1, 1) NOT NULL,
    [InvoiceIDO] INT             NOT NULL,
    [CustomerID] INT             NOT NULL,
    [DueDate]    DATE            NOT NULL,
    [IsOpen]     BIT             NOT NULL,
    [BucketID]   INT             NOT NULL,
    [OrigAmnt]   NUMERIC (16, 2) NOT NULL,
    [RemAmnt]    NUMERIC (16, 2) NOT NULL,
    [ValidFrom]  DATE            NOT NULL,
    [ValidTo]    DATE            NOT NULL,
    [IsCurrent]  BIT             NOT NULL
);


GO
PRINT N'Creating [dbo].[SCF_Invoices].[IX_IsCurrent_InvoiceIDO_InclData]...';


GO
CREATE NONCLUSTERED INDEX [IX_IsCurrent_InvoiceIDO_InclData]
    ON [dbo].[SCF_Invoices]([ValidFrom] ASC, [InvoiceID] ASC)
    INCLUDE([IsCurrent], [ValidTo], [InvoiceIDO], [CustomerID], [DueDate], [IsOpen], [BucketID], [OrigAmnt], [RemAmnt]) WHERE (IsCurrent = 1);


GO
PRINT N'Creating [dbo].[ST_Invoices]...';


GO
CREATE TABLE [dbo].[ST_Invoices] (
    [OnDate]      DATE            NOT NULL,
    [InvoiceIDO]  INT             NOT NULL,
    [CustomerIDO] INT             NOT NULL,
    [DueDate]     DATE            NOT NULL,
    [OrigAmnt]    NUMERIC (16, 2) NOT NULL,
    [RemAmnt]     NUMERIC (16, 2) NOT NULL
);


GO
PRINT N'Creating Default Constraint on [dbo].[SCF_Invoices]....';


GO
ALTER TABLE [dbo].[SCF_Invoices]
    ADD DEFAULT 1 FOR [IsOpen];


GO
PRINT N'Creating Default Constraint on [dbo].[SCF_Invoices]....';


GO
ALTER TABLE [dbo].[SCF_Invoices]
    ADD DEFAULT '99991231' FOR [ValidTo];


GO
PRINT N'Creating Default Constraint on [dbo].[SCF_Invoices]....';


GO
ALTER TABLE [dbo].[SCF_Invoices]
    ADD DEFAULT 0 FOR [IsCurrent];


GO
PRINT N'Creating FK_SCF_Invoices_Customer...';


GO
ALTER TABLE [dbo].[SCF_Invoices] WITH NOCHECK
    ADD CONSTRAINT [FK_SCF_Invoices_Customer] FOREIGN KEY ([CustomerID]) REFERENCES [dbo].[DI_Customer] ([CustomerID]);


GO
PRINT N'Creating FK_SCF_Invoices_Bucket...';


GO
ALTER TABLE [dbo].[SCF_Invoices] WITH NOCHECK
    ADD CONSTRAINT [FK_SCF_Invoices_Bucket] FOREIGN KEY ([BucketID]) REFERENCES [dbo].[DI_Bucket] ([BucketID]);


GO
PRINT N'Creating [dbo].[tvf_F_Invoices]...';


GO
CREATE FUNCTION tvf_F_Invoices(@Date date)
RETURNS TABLE
AS RETURN
( SELECT InvoiceID,
         InvoiceIDO,
         CustomerID,
         DueDate,
         IsOpen,
         BucketID,
         OrigAmnt,
         RemAmnt,
         @Date AS OnDate,
         ValidFrom,
         ValidTo,
         IsCurrent
    FROM SCF_Invoices
)
GO
PRINT N'Creating [dbo].[LOAD_SCF_Invoices]...';


GO
CREATE PROCEDURE LOAD_SCF_Invoices @LoadDate date =NULL
AS
BEGIN
  SET NOCOUNT ON

  DECLARE @Today date, @Yesterday date
  IF @LoadDate IS NULL
    SELECT @Today = MAX(OnDate) FROM ST_Invoices
  ELSE
    SET @Today=@LoadDate
  SELECT @Yesterday = DATEADD(dd, -1, @Today)

--DON'T DELETE, To avoid REFERENCE constraint conflits, instead set the Validity of the record to an imossible value
  UPDATE SCF_Invoices
     SET ValidTo = @Yesterday,
         IsCurrent =0
   WHERE ValidFrom >= @Today 

 --Correct SCF records
  UPDATE SCF_Invoices
     SET ValidTo = CASE WHEN IsOpen = 1 THEN '99991231' ELSE ValidFrom END,
         IsCurrent =1
   WHERE ValidFrom < @Today
     AND ValidTo >= @Yesterday

  -- To avoid the error: The target table 'dbo.SCF_CASE_DETAILS' of the INSERT statement cannot be on either side of a (primary key, foreign key) relationship when the FROM clause contains a nested INSERT, UPDATE, DELETE, or MERGE statement. Found reference constraint 'FK_SCF_CASE_DETAILS_CUSTOMER'.
  -- Insert in two steps. The outer insert will firs insert in a temporary table, then the tempory ry data is inserte in [dbo].[SCF_CASE_DETAILS] 
  -- Otherwise we can'thave foreign key constraints on the table, see: 
  --   http://connect.microsoft.com/SQLServer/feedback/details/435031/unable-to-perform-slowly-changing-dimension-updates-on-a-dimension-table-with-a-foreign-key
  -- Create THE temporyry table 
  CREATE TABLE #SCF_Invoices
  ( InvoiceIDO int NOT NULL,    -- Invoice ID from the ETL system
    CustomerID int NOT NULL,
    DueDate date NOT NULL,
    IsOpen bit NOT NULL DEFAULT 1,    --Indicates that the Invoice was not (fully) paid yet
    BucketID int NOT NULL,
    OrigAmnt numeric(16,2) NOT NULL,    --The original Invoice amount
    RemAmnt numeric(16,2) NOT NULL,        --The remaining, unpaid, Invoice amount
    ValidFrom date NOT NULL, 
    ValidTo DATE NOT NULL DEFAULT '99991231', 
    IsCurrent BIT NOT NULL DEFAULT 0, 
  )    
 
  -- The OUTER insert insrerts a new valid record for all records flagged as no longer active
  -- INSERT in the temporary table
  INSERT INTO #SCF_Invoices (InvoiceIDO, CustomerID, DueDate, IsOpen, BucketID, OrigAmnt, RemAmnt, ValidFrom, ValidTo, IsCurrent)
  SELECT InvoiceIDO, CustomerID, DueDate, IsOpen, BucketID, OrigAmnt, RemAmnt, @Today AS ValidFrom, CASE WHEN IsOpen=0 THEN @Today ELSE '99991231' END AS ValidTo, 1
  FROM
  ( -- Merge statement
    MERGE INTO SCF_Invoices AS DST
    USING 
      ( SELECT st.InvoiceIDO,
               ISNULL(cus.CustomerID,-1) AS CustomerID,   -- Return Unknown Customer (-1) when not matched
               st.DueDate,
               CASE WHEN st.RemAmnt > 0 THEN 1 ELSE 0 END AS IsOpen,
               ISNULL(buc.BucketID,-1) as BucketID,    --Return Unknown (-1) when not matched
               st.OrigAmnt,
               st.RemAmnt
          FROM ST_Invoices st
          LEFT JOIN DI_Customer cus
            ON cus.CustomerIDO = st.CustomerIDO
          LEFT JOIN DI_Bucket buc
            ON DATEDIFF(dd,st.DueDate,st.OnDate) BETWEEN buc.BucketFromDays AND buc.BucketToDays
         WHERE st.OnDate = @Today
      ) AS SRC
      ON (SRC.InvoiceIDO = DST.InvoiceIDO)
     AND (DST.IsCurrent = 1)    -- This conditrion limits the MERGE to Current invoice records only, and can make use of a conditional index
      -- Insert a new record if the Invoice is not already present
     WHEN NOT MATCHED THEN 
   INSERT (InvoiceIDO, CustomerID, DueDate, IsOpen, BucketID, OrigAmnt, RemAmnt, ValidFrom, ValidTo, IsCurrent)
   VALUES (SRC.InvoiceIDO, SRC.CustomerID, SRC.DueDate, SRC.IsOpen, SRC.BucketID, SRC.OrigAmnt, SRC.RemAmnt, @Today, '99991231', 1)
     -- Check if an existing Invoice record has changed data
     WHEN MATCHED 
      AND (SRC.IsOpen = 1 OR DST.IsOpen = 1) --Don't process changes on closed Invoices
      AND (DST.CustomerID <> SRC.CustomerID
	    OR DST.DueDate <> SRC.DueDate
	    OR DST.IsOpen <> SRC.IsOpen
	    OR DST.BucketID <> SRC.BucketID
	    OR DST.OrigAmnt <> SRC.OrigAmnt
	    OR DST.RemAmnt <> SRC.RemAmnt
		)
      -- When Invoice data changed, set the existing record as no longer active, and setis validity period to Yesterday
     THEN UPDATE 
      SET DST.IsCurrent = 0, 
	      DST.ValidTo = @Yesterday
    OUTPUT SRC.InvoiceIDO, SRC.CustomerID, SRC.DueDate, SRC.IsOpen, SRC.BucketID, SRC.OrigAmnt, SRC.RemAmnt, $Action AS MergeAction
  ) AS MRG
  WHERE MRG.MergeAction = 'UPDATE';  

  -- Finally insert the data from the temporary table
  INSERT INTO SCF_Invoices (InvoiceIDO, CustomerID, DueDate, IsOpen, BucketID, OrigAmnt, RemAmnt, ValidFrom, ValidTo, IsCurrent)
  SELECT InvoiceIDO, CustomerID, DueDate, IsOpen, BucketID, OrigAmnt, RemAmnt, ValidFrom, ValidTo, IsCurrent
    FROM #SCF_Invoices
END
GO
PRINT N'Creating [dbo].[SourceSelect]...';


GO
CREATE PROCEDURE SourceSelect @OnDate date 
AS
  SELECT st.InvoiceIDO,
       ISNULL(cus.CustomerID,-1) AS CustomerID,   -- Return Unknown Customer (-1) when not matched
       st.DueDate,
       CASE WHEN ISNULL(st.RemAmnt,0) > 0 THEN 1 ELSE 0 END AS IsOpen,
       ISNULL(buc.BucketID,-1) as BucketID,    --Return Unknown (-1) when not matched
       st.OrigAmnt,
       st.RemAmnt
  FROM ST_Invoices st
  LEFT JOIN DI_Customer cus
    ON cus.CustomerIDO = st.CustomerIDO
  LEFT JOIN DI_Bucket buc
    ON DATEDIFF(dd,st.DueDate,st.OnDate) BETWEEN buc.BucketFromDays AND buc.BucketToDays
 WHERE st.OnDate=@OnDate
RETURN 0
GO
PRINT N'Creating [dbo].[LOAD_ALL]...';


GO
CREATE PROCEDURE [dbo].[LOAD_ALL]
AS
BEGIN
  SET NOCOUNT ON

  DELETE FROM dbo.SCF_Invoices;

  DECLARE @LoadDate date, @LastDate date='19000101';
  SELECT @LoadDate=MIN(OnDate) FROM ST_Invoices WHERE OnDate>@LastDate;

  PRINT 'Start processing'
  WHILE @LoadDate IS NOT NULL
  BEGIN
    PRINT 'Process data for '+Cast(@LoadDate  AS VARCHAR);
    EXEC dbo.LOAD_SCF_Invoices @LoadDate=@LoadDate;

	SET @LastDate=@LoadDate;
    SELECT @LoadDate=MIN(OnDate) FROM ST_Invoices WHERE OnDate>@LastDate;
  END
  PRINT 'Finished processing'
END
GO

PRINT N'Creating [dbo].[TEST_DATA]...';


GO
CREATE PROCEDURE [dbo].[TEST_DATA]
--Compares data in SCF_Invoices day by day with the source data using EXCEPT
--Each EXCEPT should return 0 rows
AS
BEGIN

  DECLARE @LoadDate date, @LastDate date='19000101';
  SELECT @LoadDate=MIN(OnDate) FROM ST_Invoices WHERE OnDate>@LastDate;

  PRINT 'Start testing'
  WHILE @LoadDate IS NOT NULL
  BEGIN
    PRINT 'Test data for '+Cast(@LoadDate  AS VARCHAR);
    SELECT inv.OnDate,inv.InvoiceIDO,ISNULL(cus.CustomerID,-1) AS CustomerID,inv.DueDate,inv.OrigAmnt
      FROM ST_Invoices inv
      LEFT JOIN DI_Customer cus
        ON cus.CustomerIDO = inv.CustomerIDO
     WHERE OnDate=@LoadDate
    EXCEPT
    SELECT [OnDate],[InvoiceIDO],[CustomerID],[DueDate],[OrigAmnt]
      FROM dbo.tvf_F_Invoices(@LoadDate)
    
	SET @LastDate=@LoadDate;
    SELECT @LoadDate=MIN(OnDate) FROM ST_Invoices WHERE OnDate>@LastDate;
  END
  PRINT 'Finished testing'
END
GO
/*
Post-Deployment Script Template							
--------------------------------------------------------------------------------------
 This file contains SQL statements that will be appended to the build script.		
 Use SQLCMD syntax to include a file in the post-deployment script.			
 Example:      :r .\myfile.sql								
 Use SQLCMD syntax to reference a variable in the post-deployment script.		
 Example:      :setvar TableName MyTable							
               SELECT * FROM [$(TableName)]					
--------------------------------------------------------------------------------------
*/

--Delete existing data
DELETE FROM dbo.SCF_Invoices;
DELETE FROM dbo.DI_Customer;
DELETE FROM dbo.DI_Bucket;
DELETE FROM dbo.ST_Invoices;


--Insert BI_Bucket data

SET IDENTITY_INSERT [dbo].[DI_Bucket] ON
INSERT INTO [dbo].[DI_Bucket] ([BucketID], [BucketName], [BucketFromDays], [BucketToDays]) VALUES (-1, 'Error', -9999, -1)
SET IDENTITY_INSERT [dbo].[DI_Bucket] OFF
INSERT INTO [dbo].[DI_Bucket] ([BucketName], [BucketFromDays], [BucketToDays]) VALUES (N'In time', 0, 30)
INSERT INTO [dbo].[DI_Bucket] ([BucketName], [BucketFromDays], [BucketToDays]) VALUES (N'OverDue', 31, 45)
INSERT INTO [dbo].[DI_Bucket] ([BucketName], [BucketFromDays], [BucketToDays]) VALUES (N'OverDue2', 46, 60)
INSERT INTO [dbo].[DI_Bucket] ([BucketName], [BucketFromDays], [BucketToDays]) VALUES (N'OverDue3', 61, 70)
INSERT INTO [dbo].[DI_Bucket] ([BucketName], [BucketFromDays], [BucketToDays]) VALUES (N'OverDue4', 71, 80)
INSERT INTO [dbo].[DI_Bucket] ([BucketName], [BucketFromDays], [BucketToDays]) VALUES (N'Enforcement', 81, 999999)

--Insert BI_Bucket data

SET IDENTITY_INSERT [dbo].[DI_Customer] ON
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (3, 119541, N'Customer 1', N'addr1')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (4, 1117, N'Customer 2', N'addr2')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (5, 85730, N'Customer 3', N'addr3')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (6, 37196, N'Customer 4', N'addr4')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (7, 163732, N'Customer 5', N'addr5')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (8, 116821, N'Customer 6', N'addr6')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (9, 24292, N'Customer 7', N'addr7')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (10, 4270, N'Customer 8', N'addr8')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (11, 84811, N'Customer 9', N'addr9')
INSERT INTO [dbo].[DI_Customer] ([CustomerID], [CustomerIDO], [CustomerName], [CustomerAddress]) VALUES (12, 82945, N'Customer 10', N'addr10')
SET IDENTITY_INSERT [dbo].[DI_Customer] OFF


--Insert Invoice Stage data
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111213',48132075,24292,'20111205',1553.18,1052.93)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111213',48132078,24292,'20111205',325.49,40.81)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111213',48132084,24292,'20111205',2103.16,1200.96)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111216',6204384,1117,'20111216',106.68,94.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111216',50567962,37196,'20111216',49.49,49.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111216',7586237,82945,'20111216',38.00,33.99)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111216',7571293,84811,'20111216',47.51,3.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111216',50136172,85730,'20111216',9.36,9.36)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111216',6242643,116821,'20111216',40.09,32.15)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111216',50766987,119541,'20111216',46.18,46.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111220',6204384,1117,'20111216',106.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111229',50567962,37196,'20111216',49.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111229',7586237,82945,'20111216',38.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20111229',50766987,119541,'20111216',46.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120101',7571293,84811,'20111216',47.51,3.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120101',50136172,85730,'20111216',9.36,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120101',6242643,116821,'20111216',40.09,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120103',49978414,24292,'20120103',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120103',49978425,24292,'20120103',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120103',49978428,24292,'20120103',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120103',49978434,24292,'20120103',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120103',49978437,24292,'20120103',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120103',51221165,119541,'20120103',34.90,2.08)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120104',6293447,1117,'20120104',100.68,88.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120104',51283068,85730,'20120104',5.00,3.72)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120111',48132075,24292,'20111205',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120111',48132078,24292,'20111205',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120111',48132084,24292,'20111205',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120111',49978414,24292,'20120103',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120111',49978425,24292,'20120103',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120111',49978428,24292,'20120103',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120112',49978434,24292,'20120103',2103.16,2010.02)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120112',51624432,37196,'20120112',41.23,27.69)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120112',7661181,82945,'20120112',32.00,27.99)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120112',7571293,84811,'20111216',47.51,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120112',6307521,116821,'20120112',106.80,98.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120112',51624470,163732,'20120112',178.01,38.01)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120113',7633722,4270,'20120113',283.12,217.12)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120113',51624432,37196,'20120112',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120116',6375551,1117,'20120116',80.80,80.80)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120116',52241521,37196,'20120116',41.23,41.23)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120116',7796382,82945,'20120116',32.00,32.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120116',7781307,84811,'20120116',47.51,3.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120116',51801191,85730,'20120116',34.92,34.92)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120116',52440497,119541,'20120116',46.80,46.80)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120116',51795816,163732,'20120116',0.12,0.12)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120118',52241521,37196,'20120116',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120118',52614960,163732,'20120118',75.00,75.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120120',51624470,163732,'20120112',178.01,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120120',51795816,163732,'20120116',0.12,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120120',52614960,163732,'20120118',75.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120125',51221165,119541,'20120103',34.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120125',52440497,119541,'20120116',46.80,13.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120130',7666195,4270,'20120130',467.63,467.63)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120130',7816925,4270,'20120130',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120201',51283068,85730,'20120104',5.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120201',51801191,85730,'20120116',34.92,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120202',7661181,82945,'20120112',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120206',52330589,24292,'20120205',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120206',52331078,24292,'20120205',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120206',52331522,24292,'20120205',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120206',52332048,24292,'20120205',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120206',52332560,24292,'20120205',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120208',49978434,24292,'20120103',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120208',49978437,24292,'20120103',284.68,90.54)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120208',52330589,24292,'20120205',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120208',52331078,24292,'20120205',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120208',52331522,24292,'20120205',650.98,426.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120208',52332048,24292,'20120205',2103.16,1777.67)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120208',52332560,24292,'20120205',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120209',7796382,82945,'20120116',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120209',7781307,84811,'20120116',47.51,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120210',7666195,4270,'20120130',467.63,42.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120210',7816925,4270,'20120130',714.00,156.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120215',6494354,1117,'20120215',5.00,5.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120215',6560523,1117,'20120215',87.71,87.71)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120215',7633722,4270,'20120113',283.12,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120215',7666195,4270,'20120130',467.63,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120215',7816925,4270,'20120130',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120215',8014666,82945,'20120215',32.00,32.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120215',7999649,84811,'20120215',47.51,3.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120216',53674491,37196,'20120216',41.23,41.23)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120216',53238126,85730,'20120216',42.28,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120216',53873735,119541,'20120216',43.73,43.73)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120216',53232293,163732,'20120216',51.28,51.28)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120220',6443909,1117,'20120218',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120220',53674491,37196,'20120216',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120220',53232293,163732,'20120216',51.28,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120301',8033775,4270,'20120301',505.16,505.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120301',53238126,85730,'20120216',42.28,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',6293447,1117,'20120104',100.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',6375551,1117,'20120116',80.80,69.48)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',53082067,24292,'20120302',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',53082070,24292,'20120302',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',53082073,24292,'20120302',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',53082076,24292,'20120302',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',53082079,24292,'20120302',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',52440497,119541,'20120116',46.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120302',53873735,119541,'20120216',43.73,10.91)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120305',7882522,4270,'20120304',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120307',8014666,82945,'20120215',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120309',53082067,24292,'20120302',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120309',53082070,24292,'20120302',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120309',53082073,24292,'20120302',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120309',53082076,24292,'20120302',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120309',53082079,24292,'20120302',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120313',7882522,4270,'20120304',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120316',6732222,1117,'20120316',134.33,134.33)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120316',8033775,4270,'20120301',505.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120316',55140531,37196,'20120316',41.23,41.23)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120316',8215639,82945,'20120316',32.00,32.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120316',8200878,84811,'20120316',47.51,47.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120316',54709184,85730,'20120316',37.36,37.36)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120316',55336678,119541,'20120316',41.96,41.96)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120316',54699770,163732,'20120316',51.59,51.59)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120319',6443910,1117,'20120318',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120320',54699770,163732,'20120316',51.59,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120322',55140531,37196,'20120316',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120323',7999649,84811,'20120215',47.51,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120323',8200878,84811,'20120316',47.51,3.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120326',53873735,119541,'20120216',43.73,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120330',8085371,4270,'20120330',406.24,406.24)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120402',8237276,4270,'20120402',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120402',54552459,24292,'20120331',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120402',54552462,24292,'20120331',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120402',54552465,24292,'20120331',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120402',54552468,24292,'20120331',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120402',54552471,24292,'20120331',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120402',54709184,85730,'20120316',37.36,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',6375551,1117,'20120116',80.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',6443909,1117,'20120218',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',6443910,1117,'20120318',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',6494354,1117,'20120215',5.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',6560523,1117,'20120215',87.71,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',6732222,1117,'20120316',134.33,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',8085371,4270,'20120330',406.24,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',8237276,4270,'20120402',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',8215639,82945,'20120316',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',8200878,84811,'20120316',47.51,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',6307521,116821,'20120112',106.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120406',55336678,119541,'20120316',41.96,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',6375551,1117,'20120116',80.80,69.48)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',6443909,1117,'20120218',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',6443910,1117,'20120318',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',6494354,1117,'20120215',5.00,5.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',6560523,1117,'20120215',87.71,87.71)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',6732222,1117,'20120316',134.33,134.33)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',8085371,4270,'20120330',406.24,406.24)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',8237276,4270,'20120402',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',8215639,82945,'20120316',32.00,32.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',8200878,84811,'20120316',47.51,3.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',6307521,116821,'20120112',106.80,98.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120407',55336678,119541,'20120316',41.96,41.96)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120410',49978437,24292,'20120103',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120410',52331522,24292,'20120205',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120410',52332048,24292,'20120205',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120410',54552459,24292,'20120331',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120410',54552468,24292,'20120331',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120410',8215639,82945,'20120316',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120411',8237276,4270,'20120402',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120411',54552462,24292,'20120331',2103.16,700.71)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120411',54552465,24292,'20120331',650.98,366.30)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120411',54552471,24292,'20120331',1553.18,1227.69)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120412',55336678,119541,'20120316',41.96,9.14)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120416',8085371,4270,'20120330',406.24,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6375551,1117,'20120116',80.80,69.48)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6443909,1117,'20120218',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6443910,1117,'20120318',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6443911,1117,'20120418',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6494354,1117,'20120215',5.00,5.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6560523,1117,'20120215',87.71,87.71)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6732222,1117,'20120316',134.33,134.33)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6901916,1117,'20120418',74.84,74.84)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',54552462,24292,'20120331',2103.16,700.71)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',54552465,24292,'20120331',650.98,366.30)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',54552471,24292,'20120331',1553.18,1227.69)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',56526815,37196,'20120418',41.23,41.23)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',8427869,82945,'20120418',32.00,32.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',8200878,84811,'20120316',47.51,3.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',8412756,84811,'20120418',47.51,47.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',56085644,85730,'20120418',40.80,40.80)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',6307521,116821,'20120112',106.80,98.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',55336678,119541,'20120316',41.96,9.14)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120418',56725199,119541,'20120418',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120419',56526815,37196,'20120418',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120420',8412756,84811,'20120418',47.51,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120425',8427869,82945,'20120418',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120430',8294749,4270,'20120430',457.27,457.27)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120501',56085644,85730,'20120418',40.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120502',55930337,24292,'20120502',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120502',55930340,24292,'20120502',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120502',55930343,24292,'20120502',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120502',55930346,24292,'20120502',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120502',55930349,24292,'20120502',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120507',8294116,4270,'20120505',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120508',8294749,4270,'20120430',457.27,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120510',8294116,4270,'20120505',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120511',6375551,1117,'20120116',80.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120511',6494354,1117,'20120215',5.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120514',6560523,1117,'20120215',87.71,62.19)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120514',55930337,24292,'20120502',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120514',55930340,24292,'20120502',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120514',55930343,24292,'20120502',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120514',55930346,24292,'20120502',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120514',55930349,24292,'20120502',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120521',6443912,1117,'20120518',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120521',7080983,1117,'20120518',114.19,114.19)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120521',8640586,82945,'20120518',32.00,32.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120521',8625440,84811,'20120518',47.51,47.51)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120521',57407498,85730,'20120518',42.54,42.54)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120521',58047277,119541,'20120518',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120530',8506431,4270,'20120530',435.38,435.38)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120530',8200878,84811,'20120316',47.51,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120530',8625440,84811,'20120518',47.51,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120530',55336678,119541,'20120316',41.96,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120530',56725199,119541,'20120418',41.00,8.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120531',6307521,116821,'20120112',106.80,98.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120601',57407498,85730,'20120518',42.54,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120604',58242303,24292,'20120602',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120604',58242307,24292,'20120602',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120604',58242311,24292,'20120602',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120604',58242314,24292,'20120602',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120604',58242318,24292,'20120602',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120606',8660134,4270,'20120605',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120611',8660134,4270,'20120605',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120611',58242303,24292,'20120602',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120611',58242307,24292,'20120602',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120611',58242311,24292,'20120602',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120611',58242314,24292,'20120602',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120611',58242318,24292,'20120602',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120611',8640586,82945,'20120518',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120612',8506431,4270,'20120530',435.38,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120619',6443913,1117,'20120618',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120619',7255271,1117,'20120618',105.01,105.01)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120619',59278056,37196,'20120618',41.23,41.23)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120619',8849544,82945,'20120618',32.00,32.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120619',58834206,85730,'20120618',40.63,40.63)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120619',59475736,119541,'20120618',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120619',58826424,163732,'20120618',50.46,50.46)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120620',58826424,163732,'20120618',50.46,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120622',59278056,37196,'20120618',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120702',8714076,4270,'20120630',428.08,428.08)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120702',58834206,85730,'20120618',40.63,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120706',8869188,4270,'20120705',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120706',59675240,24292,'20120705',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120706',59675243,24292,'20120705',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120706',59675250,24292,'20120705',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120706',59675253,24292,'20120705',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120706',59675258,24292,'20120705',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',8714076,4270,'20120630',428.08,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',8869188,4270,'20120705',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',54552462,24292,'20120331',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',54552465,24292,'20120331',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',54552471,24292,'20120331',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',59675240,24292,'20120705',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',59675243,24292,'20120705',2103.16,2010.02)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',59675250,24292,'20120705',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',59675253,24292,'20120705',1553.18,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',59675258,24292,'20120705',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120709',56725199,119541,'20120418',41.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120710',6307521,116821,'20120112',106.80,78.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120711',8849544,82945,'20120618',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120716',6307521,116821,'20120112',106.80,78.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120719',6443914,1117,'20120718',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120719',7426680,1117,'20120718',94.26,94.26)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120719',60785663,37196,'20120718',41.23,41.23)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120719',9063139,82945,'20120718',32.00,32.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120719',60340576,85730,'20120718',41.23,41.23)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120719',60982579,119541,'20120718',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120719',60332105,163732,'20120718',51.65,51.65)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120720',60785663,37196,'20120718',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120720',60332105,163732,'20120718',51.65,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120725',9063139,82945,'20120718',32.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120731',8927096,4270,'20120730',418.86,418.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6443909,1117,'20120218',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6443910,1117,'20120318',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6443911,1117,'20120418',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6443912,1117,'20120518',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6443913,1117,'20120618',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6443914,1117,'20120718',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6560523,1117,'20120215',87.71,62.19)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6732222,1117,'20120316',134.33,134.33)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',6901916,1117,'20120418',74.84,74.84)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',7080983,1117,'20120518',114.19,114.19)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',7255271,1117,'20120618',105.01,105.01)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',7426680,1117,'20120718',94.26,94.26)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120801',60340576,85730,'20120718',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120802',60173025,24292,'20120801',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120802',60173033,24292,'20120801',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120802',60173042,24292,'20120801',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120802',60173045,24292,'20120801',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120802',60173060,24292,'20120801',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120806',8925813,4270,'20120805',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120809',8925813,4270,'20120805',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120809',60173025,24292,'20120801',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120809',60173033,24292,'20120801',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120809',60173042,24292,'20120801',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120809',60173045,24292,'20120801',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120809',60173060,24292,'20120801',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120810',6307521,116821,'20120112',106.80,58.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120814',8927096,4270,'20120730',418.86,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120816',7533864,1117,'20120815',29.00,29.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120816',62147460,37196,'20120815',41.23,41.23)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120816',62342243,119541,'20120815',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120820',6443915,1117,'20120818',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120820',7605735,1117,'20120818',89.45,89.45)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120820',62147460,37196,'20120815',41.23,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120820',9299243,82945,'20120818',35.92,35.92)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120820',61715518,85730,'20120818',41.28,41.28)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120820',61701192,163732,'20120818',54.25,54.25)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120821',9299243,82945,'20120818',35.92,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120828',61701192,163732,'20120818',54.25,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120831',9300126,4270,'20120830',397.99,397.99)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120903',61553149,24292,'20120901',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120903',61553157,24292,'20120901',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120903',61553160,24292,'20120901',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120903',61553163,24292,'20120901',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120903',61553166,24292,'20120901',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120904',61715518,85730,'20120818',41.28,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120904',6307521,116821,'20120112',106.80,38.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120906',9300327,4270,'20120905',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120911',9300126,4270,'20120830',397.99,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120911',9300327,4270,'20120905',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120911',61553149,24292,'20120901',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120911',61553157,24292,'20120901',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120911',61553160,24292,'20120901',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120911',61553163,24292,'20120901',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120911',61553166,24292,'20120901',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120918',7605735,1117,'20120818',89.45,89.45)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120919',6443916,1117,'20120918',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120919',9493397,82945,'20120918',32.90,21.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120919',63149696,85730,'20120918',40.39,40.39)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120919',63804410,119541,'20120918',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120919',63139915,163732,'20120918',53.16,53.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120920',63139915,163732,'20120918',53.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120921',6443915,1117,'20120818',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120921',6443916,1117,'20120918',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20120921',7605735,1117,'20120818',89.45,89.45)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121001',9354746,4270,'20120930',404.93,404.93)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121001',63149696,85730,'20120918',40.39,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121002',6307521,116821,'20120112',106.80,18.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121004',62979425,24292,'20121003',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121004',62979430,24292,'20121003',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121004',62979433,24292,'20121003',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121004',62979436,24292,'20121003',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121004',62979483,24292,'20121003',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121008',9516231,4270,'20121005',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121009',62979425,24292,'20121003',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121009',62979430,24292,'20121003',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121009',62979433,24292,'20121003',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121009',62979483,24292,'20121003',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121009',9493397,82945,'20120918',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121010',9516231,4270,'20121005',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121011',9354746,4270,'20120930',404.93,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121019',6443917,1117,'20121018',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121019',9708918,82945,'20121018',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121019',64615334,85730,'20121018',40.25,40.25)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121019',65271868,119541,'20121018',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',9569096,4270,'20121030',425.84,425.78)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',9708918,82945,'20121018',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',64615334,85730,'20121018',40.25,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',6307521,116821,'20120112',106.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',58047277,119541,'20120518',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',59475736,119541,'20120618',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',60982579,119541,'20120718',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',62342243,119541,'20120815',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',63804410,119541,'20120918',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121031',65271868,119541,'20121018',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121101',64441891,24292,'20121031',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121101',64441894,24292,'20121031',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121101',64441897,24292,'20121031',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121101',64441908,24292,'20121031',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121106',9568685,4270,'20121105',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121109',9568685,4270,'20121105',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121109',59675253,24292,'20120705',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121109',64441891,24292,'20121031',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121109',64441894,24292,'20121031',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121109',64441897,24292,'20121031',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121109',64441908,24292,'20121031',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121113',9569096,4270,'20121030',425.84,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121119',6443918,1117,'20121118',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121119',66517651,37196,'20121118',43.50,43.50)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121119',9934989,82945,'20121118',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121119',66056997,85730,'20121118',40.86,40.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121119',66718745,119541,'20121118',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121119',67182669,119541,'20121116',106.80,106.80)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121119',67182728,119541,'20121116',26.00,26.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121122',66517651,37196,'20121118',43.50,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121128',9934989,82945,'20121118',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121129',66718745,119541,'20121118',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121129',67182669,119541,'20121116',106.80,106.80)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121129',67182728,119541,'20121116',26.00,26.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121130',9957946,4270,'20121129',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121203',9792395,4270,'20121130',465.02,465.02)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121203',65880683,24292,'20121202',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121203',65880686,24292,'20121202',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121203',65880689,24292,'20121202',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121203',65880692,24292,'20121202',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121203',65880695,24292,'20121202',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121203',66056997,85730,'20121118',40.86,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121205',67327867,119541,'20121204',166.80,166.80)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121211',9957946,4270,'20121129',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121211',65880683,24292,'20121202',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121211',65880686,24292,'20121202',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121211',65880689,24292,'20121202',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121211',65880692,24292,'20121202',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121211',65880695,24292,'20121202',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121213',9792395,4270,'20121130',465.02,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121219',6443919,1117,'20121218',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121219',10167749,82945,'20121218',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121219',67578513,85730,'20121218',40.97,40.97)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121219',67565127,163732,'20121218',74.15,74.15)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121224',67565127,163732,'20121218',74.15,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121228',10167749,82945,'20121218',32.90,0.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121231',10020732,4270,'20121230',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121231',10187670,4270,'20121230',446.27,446.27)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20121231',67578513,85730,'20121218',40.97,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130103',67407781,24292,'20130102',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130103',67407800,24292,'20130102',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130103',67407865,24292,'20130102',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130103',67407877,24292,'20130102',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130103',67407880,24292,'20130102',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130115',67407781,24292,'20130102',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130115',67407800,24292,'20130102',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130115',67407865,24292,'20130102',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130115',67407877,24292,'20130102',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130115',67407880,24292,'20130102',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130116',10020732,4270,'20121230',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130116',10187670,4270,'20121230',446.27,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130121',6443920,1117,'20130118',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130121',10392041,82945,'20130118',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130121',69178176,85730,'20130118',39.91,39.91)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130121',69163090,163732,'20130118',435.95,435.95)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130124',10167749,82945,'20121218',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130124',10392041,82945,'20130118',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130124',69163090,163732,'20130118',435.95,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130131',10246488,4270,'20130130',621.00,621.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130131',10413297,4270,'20130130',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130201',69178176,85730,'20130118',39.91,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130204',69008248,24292,'20130202',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130204',69008254,24292,'20130202',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130204',69008326,24292,'20130202',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130204',69008331,24292,'20130202',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130204',69008337,24292,'20130202',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130212',10413297,4270,'20130130',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130212',69008248,24292,'20130202',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130212',69008254,24292,'20130202',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130212',69008326,24292,'20130202',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130212',69008331,24292,'20130202',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130212',69008337,24292,'20130202',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130219',6443921,1117,'20130218',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130219',71108174,37196,'20130218',43.50,43.50)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130219',10625224,82945,'20130218',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130219',70643111,85730,'20130218',40.12,40.12)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130222',71108174,37196,'20130218',43.50,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130301',70643111,85730,'20130218',40.12,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130304',10475489,4270,'20130302',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130304',10647320,4270,'20130302',581.40,581.40)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130304',70467465,24292,'20130303',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130304',70467468,24292,'20130303',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130304',70467471,24292,'20130303',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130304',70467474,24292,'20130303',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130304',70467479,24292,'20130303',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130304',10625224,82945,'20130218',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130311',10475489,4270,'20130302',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130311',70467465,24292,'20130303',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130311',70467468,24292,'20130303',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130311',70467471,24292,'20130303',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130311',70467474,24292,'20130303',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130311',70467479,24292,'20130303',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130315',10647320,4270,'20130302',581.40,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130319',6443922,1117,'20130318',19.00,19.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130319',72802794,37196,'20130318',43.50,43.50)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130319',10854397,82945,'20130318',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130319',72131659,85730,'20130318',38.54,38.54)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130320',72802794,37196,'20130318',43.50,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130401',10705903,4270,'20130330',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130401',10706482,4270,'20130330',807.26,651.26)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130401',71962597,24292,'20130331',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130401',71962603,24292,'20130331',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130401',71962606,24292,'20130331',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130401',71962609,24292,'20130331',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130401',71962612,24292,'20130331',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130401',72131659,85730,'20130318',38.54,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443909,1117,'20120218',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443910,1117,'20120318',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443911,1117,'20120418',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443912,1117,'20120518',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443913,1117,'20120618',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443914,1117,'20120718',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443915,1117,'20120818',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443916,1117,'20120918',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443917,1117,'20121018',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443918,1117,'20121118',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443919,1117,'20121218',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443920,1117,'20130118',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443921,1117,'20130218',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',6443922,1117,'20130318',19.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130408',9154988,1117,'20130316',475.00,475.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130410',10705903,4270,'20130330',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130410',71962597,24292,'20130331',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130410',71962603,24292,'20130331',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130410',71962606,24292,'20130331',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130410',71962609,24292,'20130331',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130410',71962612,24292,'20130331',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130410',10854397,82945,'20130318',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130415',10706482,4270,'20130330',807.26,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130419',74309409,37196,'20130418',43.50,43.50)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130419',11075004,82945,'20130418',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130419',73641381,85730,'20130418',38.47,38.47)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',6560523,1117,'20120215',87.71,62.19)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',6732222,1117,'20120316',134.33,134.33)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',6901916,1117,'20120418',74.84,74.84)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',7080983,1117,'20120518',114.19,114.19)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',7255271,1117,'20120618',105.01,105.01)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',7426680,1117,'20120718',94.26,94.26)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',7533864,1117,'20120815',29.00,29.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',7605735,1117,'20120818',89.45,89.45)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130422',9154988,1117,'20130316',475.00,475.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130423',74309409,37196,'20130418',43.50,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',6560523,1117,'20120215',87.71,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',6732222,1117,'20120316',134.33,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',6901916,1117,'20120418',74.84,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',7080983,1117,'20120518',114.19,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',7255271,1117,'20120618',105.01,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',7426680,1117,'20120718',94.26,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',7533864,1117,'20120815',29.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',7605735,1117,'20120818',89.45,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130429',9154988,1117,'20130316',475.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130501',11101924,4270,'20130430',477.18,321.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130501',11103992,4270,'20130430',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130501',73641381,85730,'20130418',38.47,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130503',73477296,24292,'20130502',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130503',73477299,24292,'20130502',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130503',73477302,24292,'20130502',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130503',73477305,24292,'20130502',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130503',73477308,24292,'20130502',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130513',11101924,4270,'20130430',477.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130513',11103992,4270,'20130430',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130513',73477296,24292,'20130502',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130513',73477299,24292,'20130502',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130513',73477302,24292,'20130502',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130513',73477305,24292,'20130502',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130513',73477308,24292,'20130502',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130513',11075004,82945,'20130418',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130520',75356301,37196,'20130518',43.50,43.50)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130520',11310899,82945,'20130518',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130520',75821632,85730,'20130518',39.89,39.89)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130520',75807033,163732,'20130518',103.86,103.86)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',10246488,4270,'20130130',621.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',59675243,24292,'20120705',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',62979436,24292,'20121003',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',75356301,37196,'20130518',43.50,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',11310899,82945,'20130518',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',75821632,85730,'20130518',39.89,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',58047277,119541,'20120518',41.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',59475736,119541,'20120618',41.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',60982579,119541,'20120718',41.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',62342243,119541,'20120815',41.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',63804410,119541,'20120918',41.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',65271868,119541,'20121018',41.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',66718745,119541,'20121118',41.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',67182669,119541,'20121116',106.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',67182728,119541,'20121116',26.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',67327867,119541,'20121204',166.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130527',75807033,163732,'20130518',103.86,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',10246488,4270,'20130130',621.00,621.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',59675243,24292,'20120705',2103.16,2010.02)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',62979436,24292,'20121003',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',11310899,82945,'20130518',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',75821632,85730,'20130518',39.89,39.89)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',58047277,119541,'20120518',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',59475736,119541,'20120618',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',60982579,119541,'20120718',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',62342243,119541,'20120815',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',63804410,119541,'20120918',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',65271868,119541,'20121018',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',66718745,119541,'20121118',41.00,41.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',67182669,119541,'20121116',106.80,106.80)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',67182728,119541,'20121116',26.00,26.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130528',67327867,119541,'20121204',166.80,166.80)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130603',11162631,4270,'20130530',424.33,268.33)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130603',67182669,119541,'20121116',106.80,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130604',75821632,85730,'20130518',39.89,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130611',11162631,4270,'20130530',424.33,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130619',77073115,37196,'20130618',43.50,43.50)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130619',11310899,82945,'20130518',32.90,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130619',11539047,82945,'20130618',32.90,32.90)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130619',76614480,85730,'20130618',46.06,46.06)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130620',77073115,37196,'20130618',43.50,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',11159513,4270,'20130530',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',11559053,4270,'20130630',75.60,75.60)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096209,24292,'20130515',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096220,24292,'20130515',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096233,24292,'20130515',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096237,24292,'20130515',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096240,24292,'20130515',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096356,24292,'20130610',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096359,24292,'20130610',1553.18,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096362,24292,'20130610',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096365,24292,'20130610',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130701',76096368,24292,'20130610',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130702',11559053,4270,'20130630',75.60,75.60)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130702',76614480,85730,'20130618',46.06,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130705',77576126,24292,'20130704',2103.16,2103.16)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130705',77576129,24292,'20130704',325.49,325.49)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130705',77576132,24292,'20130704',284.68,284.68)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130705',77576135,24292,'20130704',650.98,650.98)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130705',77576304,24292,'20130704',1553.18,1553.18)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130708',11564320,4270,'20130705',714.00,714.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130709',11564320,4270,'20130705',714.00,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130709',77576126,24292,'20130704',2103.16,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130709',77576129,24292,'20130704',325.49,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130709',77576132,24292,'20130704',284.68,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130709',77576135,24292,'20130704',650.98,0.00)
INSERT INTO [dbo].[ST_Invoices] ([OnDate], [InvoiceIDO], [CustomerIDO], [DueDate], [OrigAmnt], [RemAmnt])  VALUES('20130709',77576304,24292,'20130704',1553.18,0.00)
GO


GO
PRINT N'Checking existing data against newly created constraints';


GO
USE [SCF_Invoices];


GO
ALTER TABLE [dbo].[SCF_Invoices] WITH CHECK CHECK CONSTRAINT [FK_SCF_Invoices_Customer];

ALTER TABLE [dbo].[SCF_Invoices] WITH CHECK CHECK CONSTRAINT [FK_SCF_Invoices_Bucket];



GO
PRINT N'Update complete.';


GO
